//------------------------------------------------------------------------------
// File: BridgeDefense.cs
// Controls the bridge defense task and its vertex methods
// Author: Matthew Rudge
//------------------------------------------------------------------------------

//-Bridge Defense Helper Functions----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Activates the bridge defense task
////////////////////////////////////////////////////////////////////////////////
function startBridgeDefenseTasks()
{
   GameCamera.FlySpeed = 76;
   slgActivateTask("BridgeDefense");
   TaskModifier.strMarkVertex("BridgeDefense", "BridgeDefenseIntroDelay", $TSV_AVAIL);
   TaskModifier.strMarkVertex("BridgeDefense", "BridgeDefenseLose", $TSV_AVAIL);
}

////////////////////////////////////////////////////////////////////////////////
//! Deactivates the bridge defense task
////////////////////////////////////////////////////////////////////////////////
function endBridgeDefenseTasks()
{
   tsEndTask("BridgeDefense");
}

////////////////////////////////////////////////////////////////////////////////
//! Resets Bridge Defense Map
////////////////////////////////////////////////////////////////////////////////
function resetBridgeDefense()
{
   TaskModifier.strResetVertex("MainmapTasks", "BridgeDefenseReset");
   TaskModifier.strMarkVertex("MainmapTasks", "BridgeDefenseReset", $TSV_AVAIL);
}

////////////////////////////////////////////////////////////////////////////////
//! Rewards the player if the task is successfully completed
////////////////////////////////////////////////////////////////////////////////
function winBridgeDefense()
{
   tsModifyResource($Resource::Experience, 20, true);
   tsModifyResource($Resource::Gold, 300, true);
   tsModifyResource($Resource::Wood, 100, true);
   
   SendProductionToClient(HeroCharacter, "20 experience");
   SendProductionToClient(HeroCharacter, "300 gold");
   SendProductionToClient(HeroCharacter, "100 wood");
   
   // Quest log
   tsCompleteTaskInTracker("Cross the River", "Help Jack Cooper.");
   riverdude.deleteobject();
   
   TaskModifier.strMarkVertex("MainmapTasks", "BridgeDefenseComplete", $TSV_AVAIL);
}

////////////////////////////////////////////////////////////////////////////////
//! Creates and returns the default bandit disaster scenario for this scenario
//! \param %chance Chance of occurrence
//! \param %min Minimum number of bandits attacking
//! \param %max Maximum number of bandits attacking
////////////////////////////////////////////////////////////////////////////////
function createBridgeDefenseBanditDisasterScenario(%chance, %min, %max)
{
   %banditDisaster = new SLBanditAttackScenario() {
      chance    = %chance;
      duration  = 1000;
      
      // Amount of time to fade in/out for bandits
      fadeTime  = 1;
      
      // Gold stolen in attack
      goldMin   = 0;
      goldMax   = 0;
      goldLow   = 0;
      goldHigh  = 0;
      
      // 2 bandits to attack
      lowCount  = %min;
      highCount = %max;
      target    = "townlumbermill";
      
      // Happiness change
      happinessCount    = -25;
      happinessIncrease = 0;
      happinessPlateau  = 10;
      happinessDecrease = 0;
   };
   return %banditDisaster;
}
//------------------------------------------------------------------------------

//-BridgeDefenseIntro Vertex----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// The introduction vertex is responsible for the starting dialog, starting the 
// bandit disaster, and switching the teams of appropriate units
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseIntro::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Start introduction dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_intro", $TaskButton::More, HeroCharacter);
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_story", $TaskButton::Okay, "jack1", "bd_story");
}

function BridgeDefenseIntro::onExit(%this)
{
   
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_response", $TaskButton::Okay, HeroCharacter, "id_bridgedef_response");
   
   // Set quest and task log
   tsAddQuestToTracker(true, "Bridge Defense", "", "A group of Bandits has destroyed the bridge spanning the Rolling River. Rebuild the bridge while defending the Lumber Mill from further Bandit attacks.", "20 Experience and 200 Gold");
   tsAddTaskToTracker("Bridge Defense", "Defend the Lumber Mill.", "Make sure the Lumber Mill isn't destroyed.");
   tsAddTaskToTracker("Bridge Defense", "Bridge the River.", "Build Bridges from island to island to cross the Rolling River.");
   
   // Switch neutral team objects to player's team
   tsConvertObjTeam(jack1, $OST_PLAYER);
   tsConvertObjTeam(jack2, $OST_PLAYER);
   tsConvertObjTeam(jack3, $OST_PLAYER);
   tsSkinObj("jack1", "base");
   tsSkinObj("jack2", "base");
   tsSkinObj("jack3", "base");
   tsConvertObjTeam(townwell, $OST_PLAYER);
   tsConvertObjTeam(towncabin, $OST_PLAYER);
   tsConvertObjTeam(townwoodhut, $OST_PLAYER);
   tsConvertObjTeam(townlumbermill, $OST_PLAYER);
   tsConvertObjTeam(townfarm, $OST_PLAYER);
   tsConvertObjTeam(townshack, $OST_PLAYER);   
}
// Vertex completes after id_bridgedef_response
// move camera to start and end points for bridge. Then back to hero 
function Bridgecamerastart::onExit(%this)
{
   tsMoveCameraToPosition ("-311 -354", false);
}
// Delay time after
function Bridgecameraend::onExit(%this)
{
   tsMoveCameraToPosition ("-239 -439", false);
}
function Bridgecamerahero::onExit(%this)
{
   tsMoveCameraToObject("herocharacter", false);
}

// 1 min. Delay before Bandit disaster 
function BridgeDefensedelay::onExit(%this)
{
   
   %scenario = new SLScenario()
   {
      // update disasters only; do not change the map
      stack = true;
      
      // every 3 minutes, a disaster can occur
      addDisasterNode = "0 110 1";
      disasterStack = $DisasterStack::Push;
      
      // frequency pool, respawning time
      disasterBanditRate = "100 150";
      disasterDroughtRate = "0 100";
      disasterEarthquakeRate = "0 120";
      disasterFamineRate = "0 120";
      disasterFireRate = "0 120";
      disasterPlagueRate = "0 100";
      disasterTornadoRate = "0 100";
   };
   
   // update the bandit scenario
   %scenario.disasterBanditScenario = new SLBanditAttackScenario()
   {
      chance = 100;
      duration = 1000;
      
      fadeTime = 1;
      goldMin = 0;
      goldMax = 50;
      goldLow = 1;
      goldHigh = 5;
      lowCount = 1;
      highCount = 2;
      target    = "townlumbermill";
      
      happinessCount = -15;
      happinessIncrease = 0;
      happinessPlateau = 10;
      happinessDecrease = 0;
   };
      
   LoadScenario(%scenario);      
}
//------------------------------------------------------------------------------

//-BridgeDefenseHalfway Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is used to catch when the player reaches the midway zone and 
// increase the number of bandits in a bandit attack
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseHalfway::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.isMarkedComplete()) {
      // Notify player of halfway mark and other dialog necessities (tell them that
      // bandits have increased in number?)
      %objs = tsGetObjListFromMsg(%msg, %params);
      tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_halfway", $TaskButton::Okay, %objs.getID(0));
   }
}

function BridgeDefenseHalfway::onExit(%this)
{
   %scenario = new SLScenario()
   {
      // update disasters only; do not change the map
      stack = true;
      
      // every 3 minutes, a disaster can occur
      addDisasterNode = "0 120 1";
      disasterStack = $DisasterStack::Push;
      
      // frequency pool, respawning time
      disasterBanditRate = "100 120";
      disasterDroughtRate = "0 100";
      disasterEarthquakeRate = "0 120";
      disasterFamineRate = "0 120";
      disasterFireRate = "0 120";
      disasterPlagueRate = "0 100";
      disasterTornadoRate = "0 100";
   };
   
   // update the bandit scenario
   %scenario.disasterBanditScenario = new SLBanditAttackScenario()
   {
      chance = 100;
      duration = 1000;
      
      fadeTime = 1;
      goldMin = 0;
      goldMax = 50;
      goldLow = 1;
      goldHigh = 5;
      lowCount = 2;
      highCount = 3;
      target    = "townlumbermill";
      
      happinessCount = -15;
      happinessIncrease = 0;
      happinessPlateau = 10;
      happinessDecrease = 0;
   };
      
   LoadScenario(%scenario);   
}
//------------------------------------------------------------------------------

//-BridgeDefenseAmbush Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is used to catch when the player reaches the end zone on the
// opposite shore and spawn the final confronation
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseAmbush::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.isMarkedComplete()) {
      // Spawn bandits from barrels
      %obj = tsCreateObj(Malebandit, "meanie1", $OST_BANDIT, banditbarrel1.position, "meanie1");
      %obj.name = "Aaron Baggins";
      %char1obj = tsCreateObj(Femalebandit, "meanie2", $OST_BANDIT, banditbarrel2.position, "meanie2");
      %char1obj.name = "Fili Wineheart";
      %char2obj =tsCreateObj(Malebandit, "meanie3", $OST_BANDIT, banditbarrel3.position, "meanie3");
      %char2obj.name = "Balin Jenkins";
      %char3obj =tsCreateObj(Femalebandit, "meanie4", $OST_BANDIT, banditbarrel4.position, "meanie4");
      %char3obj.name = "Kili Wineheart";
      tsRemoveObj(banditbarrel1);
      tsRemoveObj(banditbarrel2);
      tsRemoveObj(banditbarrel3);
      tsRemoveObj(banditbarrel4);
      
      // ALEX: Modify this to what you desire
      // Turn off bandit disasters
      %scenario = new SLScenario() {
         stack = true;
         disasterBanditRate = "0 120";
      };
      %banditDisaster = createBridgeDefenseBanditDisasterScenario(0, 0, 0);

      // Update the scenario
      %scenario.disasterBanditScenario = %banditDisaster;
      LoadScenario(%scenario);     
   }
}
//------------------------------------------------------------------------------

//-BridgeDefenseDialog Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is used to wait until we know the bandit objects have been 
// created to send the dialog messages
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseDialog::onExit(%this)
{
   // Do dialog between character and bandits
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_confront", $TaskButton::Okay, "meanie1");
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_conres", $TaskButton::Okay, HeroCharacter);
}
//------------------------------------------------------------------------------

//-BridgeDefenseWin Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is used to check when the player has defeated all the spawned
// bandits from the previous vertex. On completion, dialog is launched and the
// quest hud is updated
////////////////////////////////////////////////////////////////////////////////

function BridgeDefenseWin::onExit(%this)
{
   tsSendTaskPopUpDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_win", $TaskButton::Okay, HeroCharacter);
   
   // Move Townmayor to welcome sign click dialog text
   tsMoveObjToTarget("townmayor", "-260.5 -437.5");
   tsSendTaskClickDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_mayor", $TaskButton::Okay, "townmayor", "id_bridgedef_mayor");
   
   // Close lose vertex
   TaskModifier.strMarkVertex("BridgeDefense", "BridgeDefenseLose", $TSV_CLOSED);
   
   // Update quest and task log
   tsCompleteTaskInTracker("Bridge Defense", "Defend the Lumber Mill.");
   tsCompleteTaskInTracker("Bridge Defense", "Bridge the River."); 

}
function BridgeDefenseMayor::onExit(%this)
{
   Parent::onExit(%this);
   
   tsMoveObjToTarget("townmayor", "-368 -276");
   // Launch win dialog
   tsSendTaskClickDlg("ID_TASKNAME_BRIDGEDEF", "id_bridgedef_exit", $TaskButton::Accept, HeroCharacter, "id_bridgedef_exit");
   
}
//------------------------------------------------------------------------------

//-BridgeDefenseExit Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is used to launch the player back to the main map with the
// rewards of winning this scenario
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseExitDelay::onExit(%this)
{
   Parent::onExit(%this);
   
   // Return to main map with rewards
   %scenario  = new SLScenario() {
      stack   = false;
   };
   LoadScenario(%scenario, endBridgeDefenseTasks, winBridgeDefense);
   
   // complete the bridge defense satellite
   mapsatellite_bridgedefense.disabled = false;
}
//------------------------------------------------------------------------------

//-BridgeDefenseLose Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex monitors the lose condition of the scenario, that is the 
// destruction of the lumber mill
////////////////////////////////////////////////////////////////////////////////
function BridgeDefenseLose::onExit(%this)
{
   //Move to Lumber Mill location
   tsMoveCameraToPosition ("-368 -276", false);   
}
function BridgeDefenseLosedelay::onExit(%this)
{
   Parent::onExit(%this);
   
   // Restart scenario due to lose condition
   tsRestartScenario();
}
//------------------------------------------------------------------------------

// End BridgeDefense.cs